////////////////////////////////////////////////////////////////////////////////
//  GTA Mission Assembler                                                     //
//  Copyright (c) 2002-2003  GtaScripts.org                                   //
//                                                                            //
//  This program is free software; you can redistribute it and/or modify      //
//  it under the terms of the GNU General Public License as published by      //
//  the Free Software Foundation; either version 2 of the License, or         //
//  (at your option) any later version.                                       //
//                                                                            //
//  This program is distributed in the hope that it will be useful,           //
//  but WITHOUT ANY WARRANTY; without even the implied warranty of            //
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             //
//  GNU General Public License for more details.                              //
//                                                                            //
//  You should have received a copy of the GNU General Public License         //
//  along with this program; if not, write to the Free Software               //
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

program gtama;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  Classes,
  Windows,
  GtaAssembler in 'GtaAssembler.pas';

const
  VERSION_MAJOR   = '2';
  VERSION_MINOR   = '0';
  VERSION_RELEASE = '0';
  VERSION_EXTRA   = 'Alpha';

var
  ScmAssembler: TScmAssembler;
  ScmProject: TScmProject;

////////////////////////////////////////////////////////////////////////////////

procedure DisplayInfo;
begin
  Writeln(Format('GTA Mission Assembler %s.%s.%s %s',[VERSION_MAJOR, VERSION_MINOR,
    VERSION_RELEASE, VERSION_EXTRA]));
  Writeln('Copyright (c) GtaScripts.org');
  Writeln('');
end;

procedure DisplayHelp;
begin
  Writeln('  -autoclose');
  Writeln('  -game <gta3 | gtavc>');
  Writeln('  -scm <outfile>');
  Writeln('  -opdef <opcode definition file>');
  Writeln('  -ide <object definition file>');
  Writeln('  -missdir <mission dir>');
  Writeln('  -main_thread <main thread>');
  Writeln('  -log');
  Writeln('');
end;

function ParseParams: Boolean;
var
  i: Integer;
begin
  Result := False;
  i := 1;
  if ParamCOunt = 0 then
  begin
    DisplayHelp;
    exit;
  end;

  while i <= ParamCount do
  begin

    if Pos('-autoclose', Lowercase(ParamStr(i))) = 1 then
    begin
      ScmProject.AutoClose := True;
      Inc(i);
    end else

    if Pos('-log', Lowercase(ParamStr(i))) = 1 then
    begin
      ScmProject.Log := True;
    end else

    if Pos('-opdef', Lowercase(ParamStr(i))) = 1 then
    begin
      if i = ParamCount then
      begin
        DisplayHelp;
        exit;
      end
      else
      begin
        Inc(i);
        if (ParamStr(i) = '') or (ParamStr(i)[1] = '-') then
        begin
          //DisplayHelp;
          Writeln('Error: Filename expected for -opdef argument');
          Exit;
        end;
        ScmProject.Opdef := ParamStr(i);
      end;
    end else

    if Pos('-scm', Lowercase(ParamStr(i))) = 1 then
    begin
      if i = ParamCount then
      begin
        DisplayHelp;
        exit;
      end
      else
      begin
        Inc(i);
        if (ParamStr(i) = '') or (ParamStr(i)[1] = '-') then
        begin
          //DisplayHelp;
          Writeln('Error: Filename expected for -scm argument');
          Exit;
        end;
        ScmProject.ScmFile := ParamStr(i);
      end;
    end else

    if Pos('-missdir', Lowercase(ParamStr(i))) = 1 then
    begin
      if i = ParamCount then
      begin
        DisplayHelp;
        exit;
      end
      else
      begin
        Inc(i);
        if (ParamStr(i) = '') or (ParamStr(i)[1] = '-') then
        begin
          //DisplayHelp;
          Writeln('Error: Directory expected for -missdir argument');
          Exit;
        end;
        ScmProject.MissionDir := includetrailingbackslash(ParamStr(i));
      end;
    end else

    if Pos('-ide', Lowercase(ParamStr(i))) = 1 then
    begin
      if i = ParamCount then
      begin
        DisplayHelp;
        exit;
      end
      else
      begin
        Inc(i);
        if (ParamStr(i) = '') or (ParamStr(i)[1] = '-') then
        begin
          //DisplayHelp;
          Writeln('Error: Filename expected for -ide argument');
          Exit;
        end;
        ScmProject.Ide := ParamStr(i);
      end;
    end else

    if Pos('-game', Lowercase(ParamStr(i))) = 1 then
    begin
      if i = ParamCount then
      begin
        DisplayHelp;
        exit;
      end
      else
      begin
        Inc(i);
        if (ParamStr(i) = '') or (ParamStr(i)[1] = '-') then
        begin
          //DisplayHelp;
          Writeln('Error: gta3 or gtavc expected for -game argument');
          Exit;
        end;
        if LowerCase(ParamStr(i)) = 'gta3' then
          ScmProject.Game := GTA3
        else if LowerCase(ParamStr(i)) = 'gtavc' then
          ScmProject.Game := GTAVC
        else
        begin
          Writeln('Error: gta3 or gtavc expected for -game argument');
          Exit;
        end;
      end;
    end else

    if Pos('-main_thread', Lowercase(ParamStr(i))) = 1 then
    begin
      if i = ParamCount then
      begin
        DisplayHelp;
        exit;
      end
      else
      begin
        Inc(i);
        if (ParamStr(i) = '') or (ParamStr(i)[1] = '-') then
        begin
          //DisplayHelp;
          Writeln('Error: name expected for -main_thread argument');
          Exit;
        end;
        ScmProject.MainThread := ParamStr(i);
      end;
    end else
    begin
      Writeln('Error: Unknown argument ''' + ParamStr(i) + '''');
      Exit;
    end;
    Inc(i);
  end;
  Result := True;
end;

var
  AClose: Boolean;
  begin
  DisplayInfo;

  ScmProject := TScmProject.Create;
  try
    ScmProject.AutoClose := False;
    try
      if ParseParams then
      begin
        ScmAssembler := TScmAssembler.Create(ScmProject.Game);
        try
          ScmAssembler.Assemble(ScmProject);
        finally
          ScmAssembler.Free;
        end;
      end;
    except
    end;
  finally
    AClose := ScmProject.AutoClose;
    ScmProject.Free;
  end;
  Writeln('');
  if not AClose then
  begin
    Writeln('Press [enter] to close the program...');
    Readln;
  end;
end.
