////////////////////////////////////////////////////////////////////////////////
//  GTA Mission Assembler                                                     //
//  Copyright (c) 2002-2003  GtaScripts.org                                   //
//                                                                            //
//  This program is free software; you can redistribute it and/or modify      //
//  it under the terms of the GNU General Public License as published by      //
//  the Free Software Foundation; either version 2 of the License, or         //
//  (at your option) any later version.                                       //
//                                                                            //
//  This program is distributed in the hope that it will be useful,           //
//  but WITHOUT ANY WARRANTY; without even the implied warranty of            //
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             //
//  GNU General Public License for more details.                              //
//                                                                            //
//  You should have received a copy of the GNU General Public License         //
//  along with this program; if not, write to the Free Software               //
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////
unit GtaAssembler;

interface

uses
  Classes, SysUtils, Windows, IniFiles, OpcodeLookup, IdeLookup, FastStringList,
  Labels;

const
  VERSION_MAJOR   = '2';
  VERSION_MINOR   = '0';
  VERSION_RELEASE = '0';

  // Characters
  TAB = #9;
  LF  = #10;
  CR  = #13;
  SP  = #32;

  // Sets
  LFCR   = [LF, CR];
  White  = [TAB, SP];
  Alpha  = ['A'..'Z'] + ['a'..'z'] + ['_'];
  Number = ['0'..'9'];
  AlphaNumeric = Alpha + Number;
  NumEx = Number + ['-','.'];
  Special = ['@',',','%','$','!',';'];
  SpecialEx = [';',','];

type
  TScmAssembler = class;
  TScmThread = class;

  TInstruction = class
    FOwner: TScmThread;
    FCode: TOpCodeDef;
    FNegative: Boolean;
    FParams: TList;
  public
    procedure Emit(Stream: TStream);
    function GetSize: Integer;
    constructor Create(Owner: TScmThread; Code: TOpCodeDef; Neg: Boolean);
    destructor Destroy; override;
  end;

  //////////////////////////////////////////////////////////////////////////////

  TArgument = class
  public
    function GetSize: LongWord; virtual; abstract;
    procedure Emit(Stream: TStream); virtual; abstract;
  end;

  TArgInteger = class(TArgument)
  public
    FSize: LongWord;
    Data: Integer;
  public
    constructor Create(Value: Integer);
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  TArgFloat = class(TArgument)
  public
    constructor Create(Value: Single); virtual; abstract;
  end;

  TArgFloatClass = class of TArgFloat;

  TArgFloatGTA3 = class(TArgFloat)
  public
    Data: packed record
      aType: Byte;
      Value: Smallint;
    end;
  public
    constructor Create(Value: Single); override;
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  TArgFloatVC = class(TArgFloat)
  public
    Data: packed record
       aType: Byte;
       Value: Single;
    end;
  public
    constructor Create(Value: Single); override;
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  TArgLabel = class(TArgument)
  public
    Row, Col: Integer;
    FLabel: string;
    Data: packed record
      aType: Byte;
      Value: LongInt;
    end;
  public
    constructor Create(Name: string; ARow, ACol: Integer);
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  TArgGlobalVar = class(TArgument)
  public
    FVars: TFastStringList;
    FVar: string;
    Data: packed record
      aType: Byte;
      Value: Word;
    end;
  public
    constructor Create(Name: string; Variables: TFastStringList);
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  TArgLocalVar = class(TArgument)
  public
    Data: packed record
      aType: Byte;
      Value: Word;
    end;
  public
    constructor Create(Value: Word);
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  TArgObjectRef = class(TArgument)
  private
    Row, Col: Integer;
    FName: string;
    FSize: LongWord;
    Data: Integer;
    FThread: TScmThread;
  public
    constructor Create(Name: string; Thread: TScmThread);
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  TArgMissionRef = class(TArgument)
  public
    Row, Col: Integer;
    FThread: TScmThread;
    FThreadName: string;
    Data: Integer;
    FSize: LongWord;
  public
    constructor Create(Value: string; Thread: TScmThread);
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  TArgString = class(TArgument)
  public
    Data: packed record
      Value: packed array[0..7] of Char;
    end;
  public
    constructor Create(Value: string);
    function GetSize: LongWord; override;
    procedure Emit(Stream: TStream); override;
  end;

  //////////////////////////////////////////////////////////////////////////////

  TScmThread = class
  private
    FOwner: TScmAssembler;
    FFileName: string;
    FCol, FRow: LongWord;
    FInstructions: TList;
    FLabels: TLabels;

    FOffset: LongWord;
    FCodeSize: LongWord;
  public
    constructor Create(Owner: TScmAssembler; FileName: string);
    destructor Destroy; override;
    procedure Emit(Stream: TStream);
  end;

  TGame = (GTA3, GTAVC);

  TScmProject = class
  private
    FGame: TGame;
    FAutoClose: Boolean;
    FMainThread: string;
    FMissionDir: string;
    FOutfile: string;
    FOpDefFile: string;
    FIDE: string;
    FLog: Boolean;
  public
    constructor Create;
    property Opdef: string read FOpDefFile write FOpDefFile;
    property Ide: string read FIDE write FIDE;
    property ScmFile: string read FOutfile write FOutfile;
    property MainThread: string read FMainThread write FMainThread;
    property MissionDir: string read FMissionDir write FMissionDir;
    property Game: TGame read FGame write FGame;
    property AutoClose: Boolean read FAutoClose write FAutoClose;
    property Log: Boolean read FLog write FLog;
  end;

  TScmAssembler = class
  private
    FScm: TFileStream;
    FGame: TGame;
    FProject: TScmProject;
    FThreads: TList;
    FLogFile: TextFile;
    FOpDB: TOpcodeLookup;
    FIde: TIdeLookup;

    BlankSectionSize, ObjectSectionSize, ObjectSectionOffset, ThreadSectionSize,
    ThreadSectionOffset, CodeSectionSize, CodeSectionOffset: Integer;

    FFloatArgClass: TArgFloatClass;

    procedure EmitBlankSpace;
    procedure EmitObjectNames;
    procedure EmitTempThreadPointers;
    procedure EmitThreadPointers;
    procedure ParseThread(ThreadId: Integer);
  public
    FVariables: TFastStringList;
    FObjects: TFastStringList;
    constructor Create(Game: TGame);
    destructor Destroy; override;
    procedure Assemble(Project: TScmProject);
    procedure Error(Msg: string; Thread: TScmThread; Fatal: Boolean); overload;
    procedure Error(Msg: string; Thread: TScmThread; Row, Col: Integer; Fatal: Boolean); overload;
    procedure ErrorCustom(Msg: string);
  end;

implementation

uses Math;

{ TInstruction }

constructor TInstruction.Create(Owner: TScmThread; Code: TOpCodeDef; Neg: Boolean);
begin
  FOwner := Owner;
  FNegative := Neg;
  FCode := Code;
  FParams := TList.Create;
end;

destructor TInstruction.Destroy;
var
  i: Integer;
begin
  for i:= 0 to FParams.Count -1 do
  begin
    TArgument(FParams.List[i]).Free;
  end;
  FParams.Free;
end;

procedure TInstruction.Emit(Stream: TStream);
var
  i: Integer;
  c: Word;
begin
  c := FCode.Code;
  if FNegative then
    Inc(c, $8000);
  Stream.Write(c, 2);
  for i := 0 to FParams.Count -1 do
  begin
    TArgument(FParams.List[i]).Emit(Stream);
  end;

  if FCode.NumParams = -1 then
  begin
    c := 0;
    Stream.Write(c, 1);
  end;
end;

function TInstruction.GetSize: Integer;
var
  r, i: Integer;
begin
  r := 0;
  Result := 2;
  for i := 0 to FParams.Count -1 do
    Inc(r, TArgument( FParams.List[i] ).GetSize);

  Inc(Result, r);
  if FCode.NumParams = -1 then
    Inc(Result);
end;

////////////////////////////////////////////////////////////////////////////////
// Argument Classes ////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

{ TArgInteger }

constructor TArgInteger.Create(Value: Integer);
begin
  Data := Value;
  if (Data >= -128) and (Data <= 127) then
    FSize := 2
  else if (Data >= -32768) and (Data <= 32767) then
    FSize := 3
  else
    FSize := 5;
end;

function TArgInteger.GetSize: LongWord;
begin
  Result := FSize;
end;

procedure TArgInteger.Emit(Stream: TStream);
var
  Short: packed record
    t: byte;
    v: Shortint;
  end;
  Small: packed record
    t: byte;
    v: SmallInt;
  end;
  Int: packed record
    t: byte;
    v: Longint;
  end;
begin
  if FSize = 2 then
  begin
    short.t := 04;
    short.v := Data;
    Stream.Write(Short, 2);
  end
  else if FSize = 3 then
  begin
    small.t := 05;
    small.v := Data;
    Stream.Write(Small, 3);
  end
  else
  begin
    int.t := 01;
    int.v := Data;
    Stream.Write(Int, 5);
  end;
end;

////////////////////////////////////////////////////////////////////////////////

{ TArgFloat }

constructor TArgFloatGTA3.Create(Value: Single);
begin
  Data.aType := 06;
  Data.Value := Trunc(Value * 16);
end;

function TArgFloatGTA3.GetSize: LongWord;
begin
  Result := 3;
end;

procedure TArgFloatGTA3.Emit(Stream: TStream);
begin
  Stream.Write(Data, 3);
end;

////////////////////////////////////////////////////////////////////////////////

{ TArgFloatVC }

constructor TArgFloatVC.Create(Value: Single);
begin
  Data.aType := 06;
  Data.Value := Value;
end;

function TArgFloatVC.GetSize: LongWord;
begin
  Result := 5;
end;

procedure TArgFloatVC.Emit(Stream: TStream);
begin
  Stream.Write(Data, 5);
end;

////////////////////////////////////////////////////////////////////////////////

{ TArgLabel }

constructor TArgLabel.Create(Name: string; ARow, ACol: Integer);
begin
  FLabel := Name;
  Row := ARow;
  Col := ACol;
  Data.aType := 01;
end;

function TArgLabel.GetSize: LongWord;
begin
  Result := 5;
end;

procedure TArgLabel.Emit(Stream: TStream);
begin
  Stream.Write(Data, 5);
end;

////////////////////////////////////////////////////////////////////////////////

{ TArgGlobalVar }

constructor TArgGlobalVar.Create(Name: string; Variables: TFastStringList);
begin
  FVars := Variables;
  FVar := Name;
  Data.aType := 02;
end;

function TArgGlobalVar.GetSize: LongWord;
begin
  Result := 3;
end;

procedure TArgGlobalVar.Emit(Stream: TStream);
begin
  Data.Value := 8 +  FVars.IndexOf(PChar(FVar)) * 4;
  Stream.Write(Data, 3);
end;

////////////////////////////////////////////////////////////////////////////////

{ TArgLocalVar }

constructor TArgLocalVar.Create(Value: Word);
begin
  Data.aType := 03;
  Data.Value := Value;
end;

function TArgLocalVar.GetSize: LongWord;
begin
  Result := 3;
end;

procedure TArgLocalVar.Emit(Stream: TStream);
begin
  Stream.Write(Data, 3);
end;

////////////////////////////////////////////////////////////////////////////////

{ TArgObjectRef }

constructor TArgObjectRef.Create(Name: string; Thread: TScmThread);
begin
  FName := Name;
  FThread:= Thread;
end;

function TArgObjectRef.GetSize: LongWord;
begin
  //TOOD..
  Data := FThread.FOwner.FObjects.IndexOf(PChar(FName));
  Data := -(Data+1);

  if (Data >= -128) and (Data <= 127) then
    FSize := 2
  else if (Data >= -32768) and (Data <= 32767) then
    FSize := 3
  else
    FSize := 5;
  Result := FSize;
end;

procedure TArgObjectRef.Emit(Stream: TStream);
var
  Short: packed record
    t: byte;
    v: Shortint;
  end;
  Small: packed record
    t: byte;
    v: SmallInt;
  end;
begin
  if FSize = 2 then
  begin
    short.t := 04;
    short.v := Data;
    Stream.Write(Short, 2);
  end
  else if FSize = 3 then
  begin
    small.t := 05;
    small.v := Data;
    Stream.Write(Small, 3);
  end;
  //else Add for 05, or display error
end;

////////////////////////////////////////////////////////////////////////////////

{ TArgMissionRef }

constructor TArgMissionRef.Create(Value: String; Thread: TScmThread);
begin
  Row := Thread.FRow;
  Col := Thread.FCol;
  FThread := Thread;
  FThreadName := Value;
end;

function TArgMissionRef.GetSize: LongWord;
var
  i: Integer;
begin
  Data := -1;
  for i := 1 to FThread.FOwner.FThreads.Count -1 do
  begin
    if StrIComp(PChar(TScmThread(FThread.FOwner.FThreads.Items[i]).FFileName),
      PChar(FThreadName + '.gsr')) = 0 then
    begin
      Data := i-1;
      Break;
    end;
  end;
  
  if Data < 0 then
    FThread.FOwner.Error(Format('Target thread ''%s'' is not in the project.',
      [FThreadName]), FThread, Row, Col - Length(FThreadName), True);

  if (Data >= -128) and (Data <= 127) then
    FSize := 2
  else if (Data >= -32768) and (Data <= 32767) then
    FSize := 3
  else
    FSize := 5;
  Result := FSize;
end;

procedure TArgMissionRef.Emit(Stream: TStream);
var
  Short: packed record
    t: byte;
    v: Shortint;
  end;
  Small: packed record
    t: byte;
    v: SmallInt;
  end;
begin
  if FSize = 2 then
  begin
    short.t := 04;
    short.v := Data;
    Stream.Write(Short, 2);
  end
  else if FSize = 3 then
  begin
    small.t := 05;
    small.v := Data;
    Stream.Write(Small, 3);
  end;
  //else Add for 05, or display error
end;

////////////////////////////////////////////////////////////////////////////////

{ TArgString }

constructor TArgString.Create(Value: string);
var
  Len: Integer;
begin
  Len := Length(Value) -2;
  Move(Value[2], Data.Value, Len);
  FillChar(Data.Value[Len+1], 7-Len, 0);
end;

function TArgString.GetSize: LongWord;
begin
  Result := 8;
end;

procedure TArgString.Emit(Stream: TStream);
begin
  Stream.Write(Data.Value, 8);
end;

////////////////////////////////////////////////////////////////////////////////

{ TScmThread }

constructor TScmThread.Create(Owner: TScmAssembler; FileName: string);
begin
  FOwner := Owner;
  FFileName := FileName;
  FInstructions := TList.Create;
  FLabels := TLabels.Create;
end;

destructor TScmThread.Destroy;
var
  i: Integer;
begin
  for i := 0 to FInstructions.Count -1 do
  begin
    TInstruction( FInstructions.List[i] ).Free;
  end;
  FreeAndNil(FInstructions);
  FLabels.Free;
end;

procedure TScmThread.Emit(Stream: TStream);
var i: Integer;
begin
  // Write all instructions for this thread to the stream
  for i := 0 to FInstructions.Count -1 do
    TInstruction(FInstructions.List[i]).Emit(Stream);
end;

////////////////////////////////////////////////////////////////////////////////

constructor TScmProject.Create;
begin
  FAutoClose := False;
  FLog := False;
  FMainThread := '';
  FMissionDir := '';
  FOutfile := '';
  FOpDefFile := '';
  FIDE := '';
  FGame := GTA3;
end;

////////////////////////////////////////////////////////////////////////////////

{ TScmAssembler }

constructor TScmAssembler.Create(Game: TGame);
begin
  inherited Create;
  FGame := Game;

  case FGame of
    GTA3 : FFloatArgClass := TArgFloat;
    GTAVC: FFloatArgClass := TArgFloatVC;
  else
    raise Exception.Create('Invalid game');
  end;

  DecimalSeparator := '.';
  FThreads := TList.Create;
  FVariables := TFastStringList.Create;
  FObjects := TFastStringList.Create;
  FOpDB := TOpcodeLookup.Create;
end;

destructor TScmAssembler.Destroy;
var
  i: Integer;
begin
  for i := 0 to FThreads.Count -1 do
  begin
    TScmThread(FThreads.List[i]).Free;
  end;
  FThreads.Free;
  FVariables.Free;
  FObjects.Free;
  FOpDB.Clear;
  FOpDB.Free;
  inherited;
end;

procedure TScmAssembler.EmitBlankSpace;
var
  w:word;
  b:byte;
  p: Pointer;
  dw: Longword;
begin
  // write jumpcode + offset to code + 00
  FScm.Position := 0;
  w := 2;
  b := 1;
  FScm.Write(w,2);
  FScm.Write(b,1);
  dw := 8 + FVariables.Count * 4;
  FScm.Write(dw,4);
  b := 0;
  FScm.Write(b,1);

  BlankSectionSize := dw;
  ObjectSectionOffset := dw;

  // Write blank space for the var section
  GetMem(p, FVariables.Count * 4);
  FillChar(p^, FVariables.Count * 4, 0);
  FScm.Write(p^, FVariables.Count * 4);
  FreeMem(p);
end;

procedure TScmAssembler.EmitObjectNames;
var
  w:word;
  b:byte;
  i:integer;
  s:string;
  zero: array[0..23] of Char;
begin
  ObjectSectionSize := 0;
  ObjectSectionOffset := FScm.Position;

  FScm.Seek(ObjectSectionOffset, soFromBeginning);
  ObjectSectionSize :=  12 + (FObjects.Count +1) *24;

  // write jumpcode + offset to code + 00
  w := 2;
  b := 1;
  FScm.Write(w,2);
  FScm.Write(b,1);
  i := ObjectSectionOffset + ObjectSectionSize;
  FScm.Write(i,4);
  b := 0;
  FScm.Write(b,1);

  // Write number of objects (including the first "blank" one)
  i := FObjects.Count +1;
  FScm.Write(i,4);

  zero[0]  := 'G';
  zero[1]  := 't';
  zero[2]  := 'a';
  zero[3]  := 'S';
  zero[4]  := 'c';
  zero[5]  := 'r';
  zero[6]  := 'i';
  zero[7]  := 'p';
  zero[8]  := 't';
  zero[9]  := 's';
  zero[10] := '.';
  zero[11] := 'o';
  zero[12] := 'r';
  zero[13] := 'g';
  zero[14] := '''';
  zero[15] := 's';
  zero[16] := ' ';
  zero[17] := 'M';
  zero[18] := 'A';
  zero[19] := ' ';
  zero[20] := VERSION_MAJOR;
  zero[21] := '.';
  zero[22] := VERSION_MINOR;
  zero[23] := VERSION_RELEASE;

  FScm.Write(zero, 24);
  FillChar(zero, 24, 0);
  for i := 0 to FObjects.Count -1 do
  begin
    s := FObjects.GetString(i);
    FScm.Write(s[1], Length(s));
    FScm.Write(Zero, 24-Length(s));
  end;
  ThreadSectionOffset := ObjectSectionOffset + ObjectSectionSize;
end;

procedure TScmAssembler.EmitTempThreadPointers;
begin
  ThreadSectionSize := 20 + (FThreads.Count -1) * 4;
  FScm.Size := ThreadSectionOffset + ThreadSectionSize;
  CodeSectionOffset := ThreadSectionOffset + ThreadSectionSize;
end;

procedure TScmAssembler.EmitThreadPointers;
var
  w:word;
  b:byte;
  i:integer;
begin
  FScm.Position := ThreadSectionOffset;

  // write jumpcode + offset to code + 00
  w := 2;
  b := 1;
  FScm.Write(w,2);
  FScm.Write(b,1);
  FScm.Write(CodeSectionOffset,4);
  b := 0;
  FScm.Write(b,1);

  // Write offset to mission threads
  i := TScmThread(FThreads[0]).FOffset + TScmThread(FThreads[0]).FCodeSize;
  FScm.Write(i,4);

  // Write 0
  i := TScmThread(FThreads[0]).FCodeSize;
  FScm.Write(i,4);

  // Write Threadcount
  i := FThreads.Count - 1;
  FScm.Write(i,4);

  // write thread offset array
  for i := 1 to FThreads.Count -1 do
  begin
    FScm.Write(TScmThread(FThreads[i]).FOffset, 4);
  end;
end;

////////////////////////////////////////////////////////////////////////////////

procedure TScmAssembler.ParseThread(ThreadId: Integer);
var
  TmpThread: TScmThread;

  Line, Tmp: string;
  TmpWord: Word;
  TmpFloat: Single;
  TmpInt: Integer;
  LineLen, Start: Longword;

  Neg: Boolean;

  OpDef: TOpCodeDef;
  Op: TInstruction;
  Src: TextFile;

  //////////
  CurrentParam: Integer;
  FixedParamCount: Boolean;
  //////////

  LastTokenLen: Integer;
  WantComma: Boolean;

  function GetNextParam: string;
  begin
    Result := '';
    Start := TmpThread.FCol;
    while not (Line[TmpThread.FCol] in Special+white) and (TmpThread.FCol <= LineLen) do
      Inc(TmpThread.FCol);

    if TmpThread.FCol > Start then
    begin
      LastTokenLen := TmpThread.FCol - Start;
      Result := Copy(Line, Start, LastTokenLen);
      start := TmpThread.FCol;
    end else
      LastTokenLen := 0;
  end;

  function GetNextParamEx: string;
  begin
    Result := '';
    Start := TmpThread.FCol;
    while not (Line[TmpThread.FCol] in SpecialEx+white) and (TmpThread.FCol <= LineLen) do
      Inc(TmpThread.FCol);

    if TmpThread.FCol > Start then
    begin
      LastTokenLen := TmpThread.FCol - Start;
      Result := Copy(Line, Start, TmpThread.FCol - Start);
      start := TmpThread.FCol;
    end else
      LastTokenLen := 0;
  end;
begin
  TmpThread := TScmThread(FThreads.Items[ThreadId]);

  // Open mission file
  try
    AssignFile(Src, FProject.FMissionDir + TmpThread.FFileName);
    Reset(Src);
  except
    ErrorCustom(Format('Could not open file ''%s''', [FProject.FMissionDir + TmpThread.FFileName]));
  end;

  // Init crap
  TmpThread.FCol := 1;
  TmpThread.FRow := 0;

  while not Eof(Src) do
  begin
    // Read line.
    Readln(Src, Line);
    Inc(TmpThread.FRow);
    TmpThread.FCol := 1;

    LineLen := Length(Line);

    if LineLen > 0 then
    begin
      // skip blanks
      while (Line[TmpThread.FCol] in White) and (TmpThread.FCol <= LineLen) do
        Inc(TmpThread.FCol);

      // Empty line?  //???
      if TmpThread.FCol > LineLen then
        Continue;

      // Comment
      if Line[TmpThread.FCol] = ';' then
        Continue;

      // Compiler directive
      if Line[TmpThread.FCol] = '#' then
      begin
        Inc(TmpThread.FCol);

        // Thread
        if StrLIComp(@Line[TmpThread.FCol], 'mission ', 8) = 0 then
        begin
          Inc(TmpThread.FCol, 8);

          // skip blanks
          while (Line[TmpThread.FCol] in White) and (TmpThread.FCol < LineLen) do
            Inc(TmpThread.FCol);

          // Get length of mission name
          Start := TmpThread.FCol;
          while (Line[TmpThread.FCol] in AlphaNumeric) and (TmpThread.FCol <= LineLen) do
            Inc(TmpThread.FCol);

          // Check if that mission file exists
          Tmp := Copy(Line, Start, TmpThread.FCol - Start) + '.gsr';
          if not FileExists(FProject.MissionDir + Tmp) then
            Error('File not found: ''' + Tmp + '''', TmpThread, True);

          // Create the new thread
          FThreads.Add(TScmThread.Create(Self, Tmp));
          Continue;
        end;
        Error('Invalid compiler directive', TmpThread, True);
      end;

      // Label
      if Line[TmpThread.FCol] = '@' then
      begin
        Inc(TmpThread.FCol);
        Start := TmpThread.FCol;
        while (Line[TmpThread.FCol] in AlphaNumeric) and (TmpThread.FCol <= LineLen) do
          Inc(TmpThread.FCol);

        if Start = TmpThread.FCol then
        begin
          Error('Parse error, no label name or illegal character', TmpThread, True);
        end
        else
        begin
          TmpThread.FLabels.Add(Copy(Line, Start, TmpThread.FCol - Start), TmpThread.FCol - Start, TmpThread.FInstructions.Count);
          Continue;
        end;
      end;

      // Instruction -- This should be an opcode


      // Check if opcode is negative
      Neg := False;
      if Line[TmpThread.FCol] = '!' then
      begin
        Inc(TmpThread.FCol);
        Neg := True;
      end;

      Start := TmpThread.FCol;

      while not (Line[TmpThread.FCol] in White+Special) and (TmpThread.FCol <= LineLen) do
        Inc(TmpThread.FCol);



      //OpDef := nil;
      OpDef := FOpDB.GetOpByName(Copy(Line, Start, TmpThread.FCol - Start), TmpThread.FCol - Start);
      if OpDef = nil then
      begin
         Error('Undeclared identifier: ''' + Copy(Line, Start, TmpThread.FCol - Start) + '''', TmpThread, True);
      end;

      CurrentParam := 0;
      Op := TInstruction.Create(TmpThread, OpDef, Neg);
      TmpThread.FInstructions.Add(Op);

      // Parameters

      FixedParamCount := OpDef.NumParams > -1;

      if FixedParamCount then
      begin
        Op.FParams.Capacity := OpDef.NumParams;
      end;

      WantComma := False;

      if LineLen - TmpThread.FCol > 0 then
      begin
        while TmpThread.FCol <= LineLen do
        begin

          // skip blanks
          while (Line[TmpThread.FCol] in White) and (TmpThread.FCol <= LineLen) do
            Inc(TmpThread.FCol);

          // End of line?
          if TmpThread.FCol > LineLen then
            Break;

          if Line[TmpThread.FCol] = ';' then
            Break;

          if WantComma then
          begin
            if Line[TmpThread.FCol] <> ',' then
              Error(''','' or end of line expected', TmpThread, true);
          end;

          if Line[TmpThread.FCol] = ',' then
          begin
            if WantComma = false then
              Error('Parameter expected but '','' was found', TmpThread, true);
            WantComma := False;
            Inc(TmpThread.FCol);
           // Inc(CurrentParam);
            Continue;
          end;

          WantComma := True;

          case Ord(Line[TmpThread.FCol]) of

            // Label reference
            Ord('@'):
            begin
              Inc(TmpThread.FCol);
              Tmp := GetNextParam;

              if LastTokenLen < 1 then
                Error('Label name expected', TmpThread, True);

              Op.FParams.Add(TArgLabel.Create(Tmp, TmpThread.FRow, TmpThread.FCol-1));
            end;

            // Global variable ref.
            Ord('$'):
            begin
              Inc(TmpThread.FCol);
              Tmp := GetNextParam;

              if LastTokenLen < 1 then
                Error('Variable name expected', TmpThread, True);

              FVariables.Add(PChar(tmp), LastTokenLen);
              Op.FParams.Add(TArgGlobalVar.Create(Tmp, FVariables));
            end;

            // Object ref
            Ord('%'):
            begin
              Inc(TmpThread.FCol);
              Tmp := GetNextParamEx;

              if LastTokenLen < 1 then
                Error('Object name expected', TmpThread, True);

              TmpInt := FIde.IndexOf(Tmp, LastTokenLen);
              if TmpInt < 0 then
              begin
                FObjects.Add(Tmp, LastTokenLen);
                Op.FParams.Add(TArgObjectRef.Create(Tmp, TmpThread));
              end else
              begin
                Op.FParams.Add(TArgInteger.Create(TmpInt));
              end;
            end;

            // Mission (thread) ref
            Ord('&'):
            begin
              Inc(TmpThread.FCol);
              Tmp := GetNextParam;
              if LastTokenLen < 1 then
                Error('Mission name expected', TmpThread, True);
              Op.FParams.Add(TArgMissionRef.Create(Tmp, TmpThread));
            end;

            // Local var (special)
            Ord('!'):
            begin
              Inc(TmpThread.FCol);
              Tmp := GetNextParam;

              if LastTokenLen < 1 then
                Error('Local variable name expected', TmpThread, True);
              try
                TmpWord := StrToInt('$'+Tmp);
                Op.FParams.Add(TArgLocalVar.Create(TmpWord))
              except
                on E: EConvertError do
                  Error('Not a valid local variable/counter', TmpThread, True);
              end;
            end;

            // String
            Ord('"'):
            begin
                     //TODO: Make a special GetNextStringParam?
              Tmp := GetNextParam;

              if LastTokenLen > 1 then
              begin
                if Tmp[LastTokenLen] <> '"' then
                  Error('End of string ''"'' expected', TmpThread, True);
                if LastTokenLen = 2 then
                  Error('Empty string', TmpThread, True);
              end else
                Error('End of string ''"'' expected', TmpThread, True);

              Op.FParams.Add(TArgString.Create(Tmp));
            end;

          else
          
            // Float, integer or invalid param
            Tmp := GetNextParam;
            if (Tmp[1] in NumEx) then
            begin
              if Pos('.', Tmp) > 0 then
              begin
                try
                  TmpFloat := StrToFloat(Tmp);
                  Op.FParams.Add(FFloatArgClass.Create(TmpFloat));
                except
                  Error('''' + Tmp + ''' is not a valid float value', TmpThread, True);
                end;
              end
              else
              begin
                try
                  TmpInt := StrToInt(Tmp);
                  Op.FParams.Add(TArgInteger.Create(TmpInt));
                except
                  Error('''' + Tmp + ''' is not a valid integer value', TmpThread, True);
                end;
              end;
            end
            else
            begin
              Error('Unknown argument type: ''' + Tmp + '''', TmpThread, True);
            end;
          end;

          Inc(CurrentParam);

          // Check param count
          if FixedParamCount then
          begin
            if CurrentParam > OpDef.NumParams then
              Error('Too many parameters', TmpThread, True);
          end;

        end;
        // Check param count
        if FixedParamCount then
        begin
          if CurrentParam < OpDef.NumParams then
          begin
            Error(Format('Expected %d parameters. But found %d',
              [OpDef.NumParams, CurrentParam]), TmpThread, True);
          end;
        end;
      end;
    end;
  end;
  CloseFile(Src);
end;

////////////////////////////////////////////////////////////////////////////////

procedure TScmAssembler.Assemble(Project: TScmProject);
var
  CurrThread: Integer;
  TmpThread: TScmThread;
  TimerStart: LongWord;
  i,j,k,
  CodeSize: Integer;
  lw: Integer;
  Op: TInstruction;
begin
  FScm := nil;
  FProject := Project;

  if FProject.Log then
  begin
    try
      AssignFile(FLogFile, ChangeFileExt(ParamStr(0), '.log'));
      Rewrite(FLogFile);
      //Reset(FLogFile);
    except
      WriteLn('Warning, could not open ' + ChangeFileExt(ParamStr(0), '.log'));
    end;
  end;

  if FProject.FMainThread = '' then
    FProject.FMainThread := 'MAIN.gsr';

  //if ExtractFilePath(FProject.FMainThread) <> '' then
  //  ErrorCustom('-main_thread must not contin');

  if ExtractFileName(FProject.FOutfile) = '' then
    ErrorCustom('-scm <filename> MUST be declared');

  if FProject.FMissionDir = '' then
    ErrorCustom('Mission directory not specified');

  if FProject.Ide = '' then
    ErrorCustom('IDE file not specified');

  // FIX!!
  FIde := TIdeLookup.Create;
  try
    FIde.Load(FProject.Ide);
  except
    on e: Exception do
      ErrorCustom('Could not load IDE file. ('+e.Message+')');
  end;

 //FIX!
  try
    FOpDB.Load(FProject.FOpDefFile);
  except
    ErrorCustom('Could not load opcode definition file.');
  end;


  try
    FScm := TFileStream.Create(FProject.ScmFile, fmCreate);
  except
    on E: EFCreateError do
      ErrorCustom(Format('Can''t create file ''%s''', [FProject.ScmFile]));
  end;

  try
    TimerStart := GetTickCount;
    CurrThread := -1;

    // Add main thread
    FThreads.Add(TScmThread.Create(Self, FProject.MainThread));

    // Parse source files
    while CurrThread < FThreads.Count -1 do
    begin
      Inc(CurrThread);
      ParseThread(CurrThread);
    end;

    // Write var space
    EmitBlankSpace;

    // Write objects
    EmitObjectNames;

    // Write thread pointer space (Is this really necessary?)
    EmitTempThreadPointers;

    // Do second pass, calculate label addresses. (code size)
    for i := 0 to FThreads.Count -1 do
    begin
      TmpThread := TScmThread(FThreads.List[i]);
      TmpThread.FOffset := CodeSectionSize + CodeSectionOffset;

      // Second pass - Label offsets
      k := 0;
      CodeSize := 0;
      for j := 0 to TmpThread.FInstructions.Count -1 do
      begin
        Op := TInstruction(TmpThread.FInstructions.List[j]);
        if k < TmpThread.FLabels.FLabels.Count then
        begin
          if PScmLabel(TmpThread.FLabels.FLabels[k]).Offset = j then
          begin
            PScmLabel(TmpThread.FLabels.FLabels[k]).Offset := CodeSize;
            Inc(k);
          end;
        end;
        try
          Inc(CodeSize, Op.GetSize);
        except
          //FIX
          writeln(InTtostr(i));
        end;
      end;
      TmpThread.FCodeSize := CodeSize;
      Inc(CodeSectionSize, CodeSize);
    end;

    // Third pass... convert label name to label offset
    for i := 0 to FThreads.Count -1 do
    begin
      TmpThread := TScmThread(FThreads.List[i]);
      for j := 0 to TmpThread.FInstructions.Count -1 do
      begin
        Op := TInstruction(TmpThread.FInstructions.List[j]);
        for k := 0 to Op.FParams.Count -1 do
        begin
          if TArgument(Op.FParams.Items[k]) is TArgLabel then
          begin
            lw := TmpThread.FLabels.GetOffset(PChar(TArgLabel(Op.FParams.Items[k]).FLabel));

            // Not found in current thread.. check main.
            if i <> 0 then
            begin
              if lw = -1 then
              begin
                // Check in main thread
                lw :=  TScmThread(FThreads.List[0]).FLabels.GetOffset(PChar(TArgLabel(Op.FParams.Items[k]).FLabel));
                if lw = -1 then
                begin
                  Error(
                    Format('Label ''@%s'' not declared.', [TArgLabel(Op.FParams.Items[k]).FLabel]),
                    TmpThread,
                    TArgLabel(Op.FParams.Items[k]).Row,
                    TArgLabel(Op.FParams.Items[k]).Col,
                    True);
                end;
                inc(lw, CodeSectionOffset);
              end
              else
              begin
                // Relative jump
                lw := -lw;
              end;
            end else
            begin
              if lw > -1 then
                inc(lw, CodeSectionOffset)
              else
                if lw = -1 then
                  Error(
                    Format('Label ''@%s'' not declared.', [TArgLabel(Op.FParams.Items[k]).FLabel]),
                    TmpThread,
                    TArgLabel(Op.FParams.Items[k]).Row,
                    TArgLabel(Op.FParams.Items[k]).Col,
                    True);
            end;
            TArgLabel(Op.FParams.Items[k]).Data.Value := lw;
          end;
        end;
      end;
    end;

    // Write thread pointers
    EmitThreadPointers;

    CodeSize := 0;
    // Write threads
    for i := 0 to FThreads.Count -1 do
    begin
      Inc(CodeSize, TScmThread(FThreads[i]).FInstructions.Count);
      TScmThread(FThreads[i]).Emit(FScm);
    end;

    TimerStart := GetTickCount - TimerStart;
    Writeln(Format('  Compile time: %.2f Seconds', [TimerStart / 1000]));
    Writeln('');
    Writeln('  Summary:');
    Writeln('');
    Writeln(Format('    %s %d Variable(s)', [#$FB, FVariables.Count]));
    Writeln(Format('    %s %d Local Object(s)', [#$FB, FObjects.Count]));
    Writeln(Format('    %s %d Thread(s)', [#$FB, FThreads.Count]));
    Writeln(Format('    %s %d Instruction(s)', [#$FB, CodeSize]));
    Writeln('');

    if FProject.Log then
    begin
      Writeln(FLogFile, Format('compiled,%d,%d,%d,%d,%d', [TimerStart,
        FVariables.Count, FObjects.Count, FThreads.Count,CodeSize]));
    end;

  finally
    if FProject.Log then
    begin
      try
        CloseFile(FLogFile);
      except
      end;
    end;

    // Clean up
    if Assigned(FScm) then
      FreeAndNil(FScm);
  end;
end;

procedure TScmAssembler.Error(Msg: string; Thread: TScmThread; Fatal: Boolean);
begin
  Writeln(Format('%s (%d:%d): %s', [Thread.FFileName, Thread.FRow, Thread.FCol, Msg]));

  if FProject.Log then
  try
    WriteLn(FLogFile, Format('error,%s,%d,%d,%s', [Thread.FFileName, Thread.FRow, Thread.FCol, Msg]));
  except
    WriteLn('Warning, could not write to log file');
  end;

  if Fatal then
    Abort;
end;

procedure TScmAssembler.Error(Msg: string; Thread: TScmThread; Row, Col: Integer; Fatal: Boolean);
begin
  Writeln(Format('%s (%d:%d): %s', [Thread.FFileName, Row, Col, Msg]));

  if FProject.Log then
  try
    WriteLn(FLogFile, Format('error,%s,%d,%d,%s', [Thread.FFileName, Row, Col, Msg]));
  except
    WriteLn('Warning, could not write to log file');
  end;

  if Fatal then
    Abort;
end;

procedure TScmAssembler.ErrorCustom(Msg: string);
begin
  Writeln(Msg);

  if FProject.Log then
  try
    WriteLn(FLogFile, Format('error,,,,%s', [Msg]));
  except
    WriteLn('Warning, could not write to log file');
  end;

  Abort;
end;


////////////////////////////////////////////////////////////////////////////////

end.
